<?php
/**
 * Plugin Name: Localnavi P-ins Auto Update Delay
 * Plugin URI: https://www.localnavi.biz/archives/12567.html
 * Description: プラグインの自動更新を48時間遅延させることができます
 * Version: 1.0.1
 * Author: SHINTANI Takashi
 * Author URI: https://www.localnavi.biz/
 * License: GPL v2 or later
 * Text Domain: localnavi-p-ins-auto-update-delay
 */

if (!defined('ABSPATH')) {
	exit;
}

class Plugins_Auto_Update_Delay {
	
	private $option_name = 'paud_delayed_plugins';
	private $delay_hours = 48;
	
	public function __construct() {
		// プラグイン一覧画面でのカラム追加
		add_filter('manage_plugins_columns', array($this, 'add_delay_column'));
		add_action('manage_plugins_custom_column', array($this, 'display_delay_column'), 10, 3);
		
		// プラグイン一覧画面でのアクションリンク追加
		add_filter('plugin_action_links', array($this, 'add_action_links'), 10, 4);
		
		// 一括操作の追加
		add_filter('bulk_actions-plugins', array($this, 'add_bulk_actions'));
		add_filter('handle_bulk_actions-plugins', array($this, 'handle_bulk_actions'), 10, 3);
		add_action('admin_notices', array($this, 'bulk_action_notices'));
		
		// AJAX処理
		add_action('wp_ajax_paud_toggle_delay', array($this, 'ajax_toggle_delay'));
		
		// 自動更新のフィルター
		add_filter('auto_update_plugin', array($this, 'maybe_delay_auto_update'), 10, 2);
		
		// 更新情報の記録
		add_action('set_site_transient_update_plugins', array($this, 'record_update_availability'));
		
		// スタイルとスクリプトの読み込み
		add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
		
		// デバッグ情報の表示（管理者のみ）
/*		if (defined('WP_DEBUG') && WP_DEBUG) {
			add_action('admin_notices', array($this, 'show_debug_info'));
		}*/
	}
	
	/**
	 * 自動更新が有効なプラグインのリストを取得
	 */
	private function get_auto_update_plugins() {
		// 方法1: site_option
		$auto_updates = get_site_option('auto_update_plugins', false);
		if (is_array($auto_updates)) {
			return $auto_updates;
		}
		
		// 方法2: option（シングルサイト）
		$auto_updates = get_option('auto_update_plugins', false);
		if (is_array($auto_updates)) {
			return $auto_updates;
		}

		// 方法3: WP5.5+ の wp_is_auto_update_enabled_for_type() を使用
		if (function_exists('wp_is_auto_update_enabled_for_type') 
			&& wp_is_auto_update_enabled_for_type('plugin')) {
			// 自動更新がサイト全体で有効な場合、全プラグインが対象
			if (!function_exists('get_plugins')) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}
			return array_keys(get_plugins());
		}

		return array();
	}
	
	/**
	 * デバッグ情報を表示
	 */
/*	public function show_debug_info() {
		// プラグイン画面でのみ表示
		$screen = get_current_screen();
		if (!$screen || $screen->id !== 'plugins') {
			return;
		}
		
		if (!current_user_can('manage_options')) {
			return;
		}
		
		$auto_updates = $this->get_auto_update_plugins();
		
		// 追加のデバッグ情報
		$site_option_value = get_site_option('auto_update_plugins', 'NOT_SET');
		$option_value = get_option('auto_update_plugins', 'NOT_SET');
		$is_multisite = is_multisite();
		
		$delayed_plugins = get_option($this->option_name, array());
		$update_plugins = get_site_transient('update_plugins');
		
		echo '<div class="notice notice-info" style="padding: 15px; font-family: monospace; font-size: 12px;">';
		echo '<strong>[PAUD デバッグ情報]</strong><br>';
		echo 'マルチサイト: ' . ($is_multisite ? 'はい' : 'いいえ') . '<br>';
		echo 'site_option値: ' . (is_array($site_option_value) ? esc_html(count($site_option_value)) . '個' : esc_html(var_export($site_option_value, true))) . '<br>';
		echo 'option値: ' . (is_array($option_value) ? esc_html(count($option_value)) . '個' : esc_html(var_export($option_value, true))) . '<br>';
		echo '自動更新が有効なプラグイン数: ' . esc_html(count($auto_updates)) . '<br>';
		if (!empty($auto_updates)) {
			echo '自動更新有効プラグイン: ' . esc_html(implode(', ', array_map('basename', $auto_updates))) . '<br>';
		}
		echo '遅延設定されているプラグイン数: ' . count($delayed_plugins) . '<br>';
		if (!empty($delayed_plugins)) {
			foreach ($delayed_plugins as $Plugins => $data) {
				$since = isset($data['update_available_since']) && $data['update_available_since'] ? 
						 gmdate('Y-m-d H:i:s', $data['update_available_since']) : 'まだ記録なし';
				echo '- ' . esc_html(basename($Plugins)) . ' (更新可能: ' . esc_html($since) . ')<br>';
			}
		}
		echo '更新可能なプラグイン数: ' . (isset($update_plugins->response) ? count($update_plugins->response) : 0) . '<br>';
		echo '</div>';
		// show_debug_info() の echo ブロックに追加
		$auto_update_enabled = function_exists('wp_is_auto_update_enabled_for_type') 
			? wp_is_auto_update_enabled_for_type('plugin') 
			: 'この関数は未サポート';
		echo 'サイト全体の自動更新(Plugins): ' . esc_html(var_export($auto_update_enabled, true)) . '<br>';
		echo 'WPバージョン: ' . esc_html(get_bloginfo('version')) . '<br>';
		echo 'AUTOMATIC_UPDATER_DISABLED: ' . (defined('AUTOMATIC_UPDATER_DISABLED') ? esc_html(var_export(AUTOMATIC_UPDATER_DISABLED, true)) : '未定義') . '<br>';
		echo 'WP_AUTO_UPDATE_CORE: ' . (defined('WP_AUTO_UPDATE_CORE') ? esc_html(var_export(WP_AUTO_UPDATE_CORE, true)) : '未定義') . '<br>';
	
	}*/
	/**
	 * 一括操作メニューに項目を追加
	 */
	public function add_bulk_actions($bulk_actions) {
		$bulk_actions['paud_enable_delay'] = __('自動更新遅延を有効化', 'localnavi-p-ins-auto-update-delay');
		$bulk_actions['paud_disable_delay'] = __('自動更新遅延を無効化', 'localnavi-p-ins-auto-update-delay');
		return $bulk_actions;
	}
	
	/**
	 * 一括操作の処理
	 */
	public function handle_bulk_actions($redirect_to, $action, $plugin_files) {
		if ($action !== 'paud_enable_delay' && $action !== 'paud_disable_delay') {
			return $redirect_to;
		}
		
		if (!current_user_can('update_plugins')) {
			return $redirect_to;
		}
		
		$auto_updates = $this->get_auto_update_plugins();
		$delayed_plugins = get_option($this->option_name, array());
		$update_plugins = get_site_transient('update_plugins');
		
		$processed = 0;
		$skipped = 0;
		$needs_auto_update = 0;
		
		foreach ($plugin_files as $plugin_file) {
			$is_auto_update_enabled = in_array($plugin_file, $auto_updates, true);
			
			if ($action === 'paud_enable_delay') {
				// 遅延を有効化
				if (!isset($delayed_plugins[$plugin_file])) {
					$delayed_plugins[$plugin_file] = array(
						'enabled_at' => time(),
						'update_available_since' => null
					);
					
					// 既に更新が利用可能な場合は記録
					if (isset($update_plugins->response[$plugin_file])) {
						$delayed_plugins[$plugin_file]['update_available_since'] = time();
					}
					
					$processed++;
					
					// 自動更新が無効の場合はカウント
					if (!$is_auto_update_enabled) {
						$needs_auto_update++;
					}
				}
			} elseif ($action === 'paud_disable_delay') {
				// 遅延を無効化
				if (isset($delayed_plugins[$plugin_file])) {
					unset($delayed_plugins[$plugin_file]);
					$processed++;
				}
			}
		}
		
		update_option($this->option_name, $delayed_plugins);
		
		$redirect_to = add_query_arg(array(
			'paud_bulk_action' => $action,
			'paud_processed' => $processed,
			'paud_skipped' => $skipped,
			'paud_needs_auto_update' => $needs_auto_update,
			'paud_bulk_nonce' => wp_create_nonce('paud_bulk_action')
		), $redirect_to);
		
		return $redirect_to;
	}
	
	/**
	 * 一括操作の完了通知
	 */
	public function bulk_action_notices() {
		if (!isset($_GET['paud_bulk_action'])) {
			return;
		}

		// nonce検証
		if (!isset($_GET['paud_bulk_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['paud_bulk_nonce'])), 'paud_bulk_action')) {
			wp_die(esc_html__('セキュリティチェックに失敗しました。', 'localnavi-p-ins-auto-update-delay'));
		}

		$action = sanitize_text_field(wp_unslash($_GET['paud_bulk_action']));
		$processed = isset($_GET['paud_processed']) ? intval($_GET['paud_processed']) : 0;
		$skipped = isset($_GET['paud_skipped']) ? intval($_GET['paud_skipped']) : 0;
		$needs_auto_update = isset($_GET['paud_needs_auto_update']) ? intval($_GET['paud_needs_auto_update']) : 0;
		
		if ($action === 'paud_enable_delay') {
			$message = sprintf(
				/* translators: %d: Number of plugins with auto-update enabled. */
				__('%d件のプラグインで自動更新遅延を有効化しました。', 'localnavi-p-ins-auto-update-delay'),
				$processed
			);
			
			if ($needs_auto_update > 0) {
				$message .= sprintf(
					/* translators: %d: Number of plugins with auto-updates disabled. */
					__('<br><strong>注意:</strong> %d件のプラグインは自動更新が無効です。遅延を機能させるには、各プラグインの「自動更新を有効化」をクリックしてください。', 'localnavi-p-ins-auto-update-delay'),
					$needs_auto_update
				);
			}
		} elseif ($action === 'paud_disable_delay') {
			$message = sprintf(
				/* translators: %d: Number of plugins with auto-update delay disabled. */
				__('%d件のプラグインで自動更新遅延を無効化しました。', 'localnavi-p-ins-auto-update-delay'),
				$processed
			);
		} else {
			return;
		}
		
		if ($skipped > 0) {
			$message .= sprintf(
				/* translators: %d: Number of plugins that skipped the auto-update delay. */
				__(' (%d件はスキップしました)', 'localnavi-p-ins-auto-update-delay'),
				$skipped
			);
		}
		
		$notice_class = ($needs_auto_update > 0) ? 'notice-warning' : 'notice-success';
		
		printf(
			'<div class="notice %s is-dismissible"><p>%s</p></div>',
			esc_attr($notice_class),
			esc_html($message)
		);
	}
	
	/**
	 * プラグイン一覧にカラムを追加
	 */
	public function add_delay_column($columns) {
		$columns['auto_update_delay'] = __('自動更新遅延', 'localnavi-p-ins-auto-update-delay');
		return $columns;
	}
	
	/**
	 * 遅延カラムの内容を表示
	 */
	public function display_delay_column($column_name, $plugin_file, $plugin_data) {
		if ($column_name !== 'auto_update_delay') {
			return;
		}
		
		// プラグインの自動更新が有効かチェック
		$auto_updates = $this->get_auto_update_plugins();
		
		// デバッグ: 最初の実行時のみログ出力
/*		static $debug_logged = false;
		if (!$debug_logged && defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
			error_log('[PAUD] Auto-update plugins count: ' . count($auto_updates));
			error_log('[PAUD] Auto-update plugins: ' . print_r($auto_updates, true));
			error_log('[PAUD] Current plugin: ' . $plugin_file);
			error_log('[PAUD] Is multisite: ' . (is_multisite() ? 'YES' : 'NO'));
			error_log('[PAUD] site_option: ' . var_export(get_site_option('auto_update_plugins', 'NOT_SET'), true));
			error_log('[PAUD] option: ' . var_export(get_option('auto_update_plugins', 'NOT_SET'), true));
			$debug_logged = true;
		}*/
		
		$is_auto_update_enabled = in_array($plugin_file, $auto_updates, true);
		$delayed_plugins = get_option($this->option_name, array());
		$is_delayed = isset($delayed_plugins[$plugin_file]);
		
		// デバッグ: このプラグインの状態をログ出力
/*		if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
			error_log(sprintf(
				'[PAUD] Plugins: %s, Auto-update: %s, Delayed: %s',
				$plugin_file,
				$is_auto_update_enabled ? 'YES' : 'NO',
				$is_delayed ? 'YES' : 'NO'
			));
		}*/
		
		// 自動更新が無効の場合
		if (!$is_auto_update_enabled) {
			if ($is_delayed) {
				// 遅延設定はあるが自動更新が無効
				echo '<span style="color: #d63638;">要自動更新</span>';
				echo '<br><small style="color: #666;">自動更新を有効化してください</small>';
			} else {
				// 自動更新も遅延も無効
				echo '<span style="color: #999;">—</span>';
			}
			return;
		}
		
		// 自動更新が有効の場合
		$status = $is_delayed ? '有効' : '無効';
		$class = $is_delayed ? 'paud-enabled' : 'paud-disabled';
		
		echo '<span class="' . esc_attr($class) . '">' . esc_html($status) . '</span>';
		
		if ($is_delayed && isset($delayed_plugins[$plugin_file]['update_available_since'])) {
			$available_since = $delayed_plugins[$plugin_file]['update_available_since'];
			$delay_until = $available_since + ($this->delay_hours * HOUR_IN_SECONDS);
			$remaining = $delay_until - time();
			
			if ($remaining > 0) {
				$hours_remaining = ceil($remaining / HOUR_IN_SECONDS);
				echo '<br><small style="color: #666;">残り約' . esc_html($hours_remaining) . '時間</small>';
			} else {
				echo '<br><small style="color: #0a0;">更新可能</small>';
			}
		}
	}
	
	/**
	 * プラグインアクションリンクを追加
	 */
	public function add_action_links($actions, $plugin_file, $plugin_data, $context) {
		// プラグインの自動更新が有効かチェック
		$auto_updates = $this->get_auto_update_plugins();
		$is_auto_update_enabled = in_array($plugin_file, $auto_updates, true);
		
		$delayed_plugins = get_option($this->option_name, array());
		$is_delayed = isset($delayed_plugins[$plugin_file]);
		
		// 自動更新が無効で遅延設定もない場合は、リンクを表示しない
		if (!$is_auto_update_enabled && !$is_delayed) {
			return $actions;
		}
		
		// 自動更新が無効だが遅延設定がある場合は、解除リンクのみ表示
		if (!$is_auto_update_enabled && $is_delayed) {
			$text = '遅延解除';
			$nonce = wp_create_nonce('paud_toggle_delay_' . $plugin_file);
			
			$link = sprintf(
				'<a href="#" class="paud-toggle-delay" data-plugin="%s" data-nonce="%s" style="color: #d63638;">%s</a>',
				esc_attr($plugin_file),
				esc_attr($nonce),
				esc_html($text)
			);
			
			$actions['auto_update_delay'] = $link;
			return $actions;
		}
		
		// 自動更新が有効な場合は、通常のトグルリンクを表示
		$text = $is_delayed ? '遅延解除' : '遅延設定';
		$nonce = wp_create_nonce('paud_toggle_delay_' . $plugin_file);
		
		$link = sprintf(
			'<a href="#" class="paud-toggle-delay" data-plugin="%s" data-nonce="%s">%s</a>',
			esc_attr($plugin_file),
			esc_attr($nonce),
			esc_html($text)
		);
		
		$actions['auto_update_delay'] = $link;
		
		return $actions;
	}
	
	/**
	 * AJAX: 遅延設定の切り替え
	 */
	public function ajax_toggle_delay() {
		$plugin_file = isset($_POST['plugin']) ? sanitize_text_field(wp_unslash($_POST['plugin'])) : '';
		check_ajax_referer('paud_toggle_delay_' . $plugin_file, 'nonce');
		
		if (!current_user_can('update_plugins')) {
			wp_send_json_error('権限がありません');
		}
		
		$delayed_plugins = get_option($this->option_name, array());
		
		if (isset($delayed_plugins[$plugin_file])) {
			// 遅延解除
			unset($delayed_plugins[$plugin_file]);
			$message = '自動更新の遅延を解除しました';
			$new_status = false;
		} else {
			// 遅延設定
			$delayed_plugins[$plugin_file] = array(
				'enabled_at' => time(),
				'update_available_since' => null
			);
			
			// 既に更新が利用可能な場合は記録
			$update_plugins = get_site_transient('update_plugins');
			if (isset($update_plugins->response[$plugin_file])) {
				$delayed_plugins[$plugin_file]['update_available_since'] = time();
			}
			
			$message = '自動更新を48時間遅延させるように設定しました';
			$new_status = true;
		}
		
		update_option($this->option_name, $delayed_plugins);
		
		wp_send_json_success(array(
			'message' => $message,
			'delayed' => $new_status
		));
	}
	
	/**
	 * 更新情報が利用可能になった時刻を記録
	 */
	public function record_update_availability($value) {
		// transientの値が不正な場合
		if (!is_object($value)) {
			return $value;
		}
		
		if (!isset($value->response) || empty($value->response)) {
			return $value;
		}
		
		$delayed_plugins = get_option($this->option_name, array());
		
		// 遅延設定されているプラグインがない場合は何もしない
		if (empty($delayed_plugins)) {
			return $value;
		}
		
		$updated = false;
		
		foreach ($delayed_plugins as $plugin_file => $data) {
			// データの整合性チェック
			if (!is_array($data)) {
				continue;
			}
			
			// 更新が利用可能で、まだ記録していない場合（nullまたは未設定または空文字）
			if (isset($value->response[$plugin_file])) {
				$needs_recording = !array_key_exists('update_available_since', $data) || 
								   $data['update_available_since'] === null || 
								   $data['update_available_since'] === '' ||
								   $data['update_available_since'] === 0;
				
				if ($needs_recording) {
					$delayed_plugins[$plugin_file]['update_available_since'] = time();
					$updated = true;
					
/*					if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
						error_log(sprintf(
							'[PAUD] Recorded update availability for plugin: %s at %s',
							$plugin_file,
							gmdate('Y-m-d H:i:s')
						));
					}*/
				}
			} 
			// 更新が利用不可になった場合はリセット
			else if (array_key_exists('update_available_since', $data) && 
					 $data['update_available_since'] !== null && 
					 $data['update_available_since'] !== '') {
				$delayed_plugins[$plugin_file]['update_available_since'] = null;
				$updated = true;
				
/*				if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
					error_log(sprintf(
						'[PAUD] Reset update availability for plugin: %s (update no longer available)',
						$plugin_file
					));
				}*/
			}
		}
		
		if ($updated) {
			update_option($this->option_name, $delayed_plugins);
		}
		
		return $value;
	}
	
	/**
	 * 自動更新を遅延させるかどうかを判定
	 */
	public function maybe_delay_auto_update($update, $item) {
		// $itemがオブジェクトでない、またはpluginプロパティがない場合
		if (!is_object($item) || !isset($item->plugin)) {
			return $update;
		}
		
		$plugin_file = $item->plugin;
		$delayed_plugins = get_option($this->option_name, array());
		
		// デバッグログ（必要に応じて）
/*		if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
			error_log(sprintf(
				'[PAUD] Checking plugin: %s, Original $update value: %s',
				$plugin_file,
				var_export($update, true)
			));
		}*/
		
		// 遅延設定がない場合は通常通り
		if (!isset($delayed_plugins[$plugin_file])) {
			return $update;
		}
		
		// 更新が利用可能になった時刻がない場合は通常通り更新を許可
		// (まだ更新情報が記録されていない状態)
		if (!array_key_exists('update_available_since', $delayed_plugins[$plugin_file]) || 
			$delayed_plugins[$plugin_file]['update_available_since'] === null || 
			$delayed_plugins[$plugin_file]['update_available_since'] === '') {
			
/*			if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
				error_log(sprintf(
					'[PAUD] Plugin %s: No update timestamp recorded yet, allowing update',
					$plugin_file
				));
			}*/
			return $update;
		}
		
		$available_since = intval($delayed_plugins[$plugin_file]['update_available_since']);
		$delay_until = $available_since + ($this->delay_hours * HOUR_IN_SECONDS);
		$current_time = time();
		
/*		if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
			error_log(sprintf(
				'[PAUD] Plugin %s: Available since: %s, Delay until: %s, Current: %s',
				$plugin_file,
				gmdate('Y-m-d H:i:s', $available_since),
				gmdate('Y-m-d H:i:s', $delay_until),
				gmdate('Y-m-d H:i:s', $current_time)
			));
		}*/
		
		// 遅延期間が過ぎていれば更新を許可
		if ($current_time >= $delay_until) {
/*			if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
				error_log(sprintf('[PAUD] Plugin %s: Delay period passed, allowing update', $plugin_file));
			}*/
			return $update;
		}
		
		// まだ遅延期間中なので更新しない
/*		if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
			$hours_remaining = ceil(($delay_until - $current_time) / HOUR_IN_SECONDS);
			error_log(sprintf(
				'[PAUD] Plugin %s: Still in delay period (%d hours remaining), blocking update',
				$plugin_file,
				$hours_remaining
			));
		}*/
		return false;
	}
	
	/**
	 * 管理画面用のアセット読み込み
	 */
	public function enqueue_admin_assets($hook) {
		if ($hook !== 'plugins.php') {
			return;
		}
		
		wp_add_inline_style('wp-admin', '
			.paud-enabled { color: #0a0; font-weight: 600; }
			.paud-disabled { color: #999; }
			.paud-toggle-delay { text-decoration: none; }
		');
		
		wp_add_inline_script('jquery', '
			jQuery(document).ready(function($) {
				$(document).on("click", ".paud-toggle-delay", function(e) {
					e.preventDefault();
					
					var $link = $(this);
					var plugin = $link.data("plugin");
					var nonce = $link.data("nonce");
					
					$.ajax({
						url: ajaxurl,
						type: "POST",
						data: {
							action: "paud_toggle_delay",
							plugin: plugin,
							nonce: nonce
						},
						success: function(response) {
							if (response.success) {
								alert(response.data.message);
								location.reload();
							} else {
								alert("エラー: " + response.data);
							}
						}
					});
				});
			});
		');
	}
}

// プラグインの初期化
$paud_instance = new Plugins_Auto_Update_Delay();

// 有効化時に既存の更新情報をスキャンして記録
register_activation_hook(__FILE__, function() {
    $option_name    = 'paud_delayed_plugins';
    $delayed_plugins = get_option($option_name, array());

    if (empty($delayed_plugins)) return;

    $update_plugins = get_site_transient('update_plugins');

    if (!isset($update_plugins->response)) return;

    foreach ($delayed_plugins as $plugin_file => $data) {
        if (isset($update_plugins->response[$plugin_file])
            && empty($data['update_available_since'])) {
            $delayed_plugins[$plugin_file]['update_available_since'] = time();
        }
    }
    update_option($option_name, $delayed_plugins);
});

require 'plugin-update-checker/plugin-update-checker.php';
use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

$myUpdateChecker = PucFactory::buildUpdateChecker(
	'https://github.com/localnavi-biz/localnavi-p-ins-auto-update-delay/',
	__FILE__,
	'localnavi-p-ins-auto-update-delay.php'
);

//Set the branch that contains the stable release.
$myUpdateChecker->setBranch('main');

//Optional: If you're using a private repository, specify the access token like this:
$myUpdateChecker->setAuthentication('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');
